<?php

declare(strict_types=1);

namespace Core\Framework\Http\Controller\Api;

use Core\General\Filter\BaseFilter;
use Core\General\Pagination\ValueObject\PaginationData;
use Core\General\Translation\Service\Translator;
use Core\General\Validation\ValueObject\ValidationResult;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

abstract class BaseController extends AbstractController
{
    public function __construct(
        protected readonly Translator $translator)
    {
    }

    protected function extractFilter(Request $request): BaseFilter
    {
        $filterQueryParam = $request->get('s');
        $orderByQueryParams = $request->get('orderBy');

        $filter = \is_string($filterQueryParam) ? $filterQueryParam : null;
        $orderBy = \is_array($orderByQueryParams) ? $orderByQueryParams : [];

        return new BaseFilter($filter, $orderBy);
    }

    protected function formatValidationErrors(ValidationResult $validationResult): JsonResponse
    {
        $message = $validationResult->getCustomErrorMessage() ?? 'exception.unprocessable_content';

        return $this->returnJsonResponse([
            'error' => $this->translator->translateMessage($message),
            'validations' => $this->translator->translateMessage($validationResult->getErrorMap()),
        ], Response::HTTP_UNPROCESSABLE_ENTITY);
    }

    /**
     * @return array<string, mixed>
     *
     * @throws \JsonException
     */
    protected function extractJSONPayload(Request $request): array
    {
        return json_decode((string) $request->getContent(), true, 512, \JSON_THROW_ON_ERROR);
    }

    protected function extractPaginationData(Request $request): PaginationData
    {
        $limit = $request->get('limit');
        $offset = $request->get('offset');

        $limit = is_numeric($limit) && (int) $limit >= 0 ? (int) $limit : null;
        $offset = is_numeric($offset) && (int) $offset >= 0 ? (int) $offset : 0;

        return new PaginationData($limit, $offset);
    }

    /**
     * @param array<string, mixed> $headers
     */
    protected function returnJsonResponse(mixed $data = null, int $status = 200, array $headers = [], bool $isDataAlreadyJson = false): JsonResponse
    {
        return new JsonResponse($data, $status, $headers, $isDataAlreadyJson);
    }

    /**
     * @param array<string, mixed> $headers
     */
    protected function returnOkResponse(mixed $data = null, array $headers = [], bool $isDataAlreadyJson = false): JsonResponse
    {
        return $this->returnJsonResponse($data, Response::HTTP_OK, $headers, $isDataAlreadyJson);
    }

    /**
     * @param array<string, mixed> $headers
     */
    protected function returnCreatedResponse(mixed $data = null, array $headers = [], bool $isDataAlreadyJson = false): JsonResponse
    {
        return $this->returnJsonResponse($data, Response::HTTP_CREATED, $headers, $isDataAlreadyJson);
    }

    /**
     * @param array<string, mixed> $headers
     */
    protected function returnNoContentResponse(mixed $data = null, array $headers = [], bool $isDataAlreadyJson = false): JsonResponse
    {
        return $this->returnJsonResponse($data, Response::HTTP_NO_CONTENT, $headers, $isDataAlreadyJson);
    }

    /**
     * @param array<string, mixed> $headers
     */
    protected function returnPaginatedResponse(mixed $data, int $itemsTotalCount, array $headers = [], bool $isDataAlreadyJson = false): JsonResponse
    {
        $headers['Total-Count'] = $itemsTotalCount;

        return $this->returnJsonResponse($data, Response::HTTP_OK, $headers, $isDataAlreadyJson);
    }
}
